<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\BankAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BankAccountController extends Controller
{
    public function index()
    {
        $accounts = BankAccount::orderBy('is_default', 'desc')->get();
        return response()->json($accounts);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'bank_name' => 'required|string',
            'account_name' => 'required|string',
            'account_number' => 'required|string',
            'ifsc_code' => 'required|string',
            'upi_id' => 'required|string',
            'is_default' => 'boolean',
        ]);

        return DB::transaction(function () use ($validated) {
            if ($validated['is_default'] ?? false) {
                BankAccount::where('is_default', true)->update(['is_default' => false]);
            }

            $account = BankAccount::create($validated);
            return response()->json($account, 201);
        });
    }

    public function update(Request $request, $id)
    {
        $account = BankAccount::findOrFail($id);

        $validated = $request->validate([
            'bank_name' => 'required|string',
            'account_name' => 'required|string',
            'account_number' => 'required|string',
            'ifsc_code' => 'required|string',
            'upi_id' => 'required|string',
            'is_default' => 'boolean',
        ]);

        return DB::transaction(function () use ($account, $validated) {
            if ($validated['is_default'] ?? false) {
                // If setting this to default, unset others, unless it was already default
                BankAccount::where('id', '!=', $account->id)->where('is_default', true)->update(['is_default' => false]);
            }

            $account->update($validated);
            return response()->json($account);
        });
    }

    public function makeDefault($id)
    {
        $account = BankAccount::findOrFail($id);

        return DB::transaction(function () use ($account) {
            // Unset all other defaults explicitly
            BankAccount::where('id', '!=', $account->id)->update(['is_default' => false]);

            // Set this one as default
            $account->is_default = true;
            $account->save();

            return response()->json(['message' => 'Default bank account updated', 'account' => $account]);
        });
    }

    public function destroy($id)
    {
        $account = BankAccount::findOrFail($id);
        $account->delete();
        return response()->json(['message' => 'Deleted successfully']);
    }
}
