<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function stats(Request $request)
    {
        // 1. Parse Filters
        $fromDate = $request->input('from_date', Carbon::now()->startOfMonth()->toDateString());
        $toDate = $request->input('to_date', Carbon::now()->endOfMonth()->toDateString());
        $brandId = $request->input('brand_id');
        $sizeId = $request->input('size_id');
        $subSizeId = $request->input('sub_size_id');
        $supplierId = $request->input('supplier_id');
        $viewBy = $request->input('view_by', 'day'); // day, month, year

        // Helper to apply common filters to queries
        $applyFilters = function ($query, $joinTable = 'stock_batches', $batchIdCol = 'batch_id') use ($brandId, $sizeId, $subSizeId, $supplierId) {
            if ($brandId && $brandId !== 'ALL') {
                $query->where("$joinTable.brand_id", $brandId);
            }
            if ($sizeId && $sizeId !== 'ALL') {
                $query->where("$joinTable.size_id", $sizeId);
            }
            if ($subSizeId && $subSizeId !== 'ALL') {
                $query->where("$joinTable.sub_size_id", $subSizeId);
            }
            if ($supplierId && $supplierId !== 'ALL') {
                $query->where("$joinTable.source_id", $supplierId);
            }
        };

        // ------------------------------------------------------------------
        // A. Sales Analysis (Revenue, COGS, Sold Qty, Profit)
        // ------------------------------------------------------------------
        // We need to join sales_order_items with stock_batches to get cost
        $salesQuery = DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('stock_batches', 'sales_order_items.batch_id', '=', 'stock_batches.id')
            ->whereBetween('sales_orders.invoice_date', [$fromDate, $toDate])
            ->where('sales_orders.status', '!=', 'CANCELLED');

        $applyFilters($salesQuery, 'stock_batches');

        // Aggregate Sales Data
        $salesStats = $salesQuery->select(
            DB::raw('SUM(sales_order_items.quantity_pieces) as total_sold_pieces'),
            DB::raw('SUM(sales_order_items.quantity_pieces * sales_order_items.price_per_piece) as total_revenue'),
            DB::raw('SUM(sales_order_items.quantity_pieces * stock_batches.cost_per_piece) as total_cogs'),
            DB::raw('COUNT(DISTINCT sales_orders.id) as total_orders')
        )->first();

        // ------------------------------------------------------------------
        // B. Returns Analysis
        // ------------------------------------------------------------------
        $returnsQuery = DB::table('sales_returns')
            ->join('stock_batches', 'sales_returns.batch_id', '=', 'stock_batches.id')
            ->whereBetween('sales_returns.created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"]);

        $applyFilters($returnsQuery, 'stock_batches');

        $returnsStats = $returnsQuery->select(
            DB::raw('SUM(sales_returns.quantity_pieces) as total_returned_pieces'),
            // Return value is based on the rate given at return time (price_per_piece)
            DB::raw('SUM(sales_returns.quantity_pieces * sales_returns.price_per_piece) as total_return_value')
        )->first();

        // ------------------------------------------------------------------
        // C. Damages Analysis
        // ------------------------------------------------------------------
        $damagesQuery = DB::table('sales_damages')
            ->join('stock_batches', 'sales_damages.batch_id', '=', 'stock_batches.id')
            ->whereBetween('sales_damages.created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"]);

        $applyFilters($damagesQuery, 'stock_batches');

        $damagesStats = $damagesQuery->select(
            DB::raw('SUM(sales_damages.quantity_pieces) as total_damaged_pieces'),
            // Damage value is Cost Price
            DB::raw('SUM(sales_damages.quantity_pieces * stock_batches.cost_per_piece) as total_damage_value')
        )->first();


        // D. Calculate Summary (Updated with Expenses)
        // ------------------------------------------------------------------
        $expensesQuery = DB::table('expenses')
            ->whereBetween('expense_date', [$fromDate, $toDate])
            ->whereNull('deleted_at')
            ->where('is_void', false);

        $totalExpenses = $expensesQuery->sum('amount');

        $revenue = $salesStats->total_revenue ?? 0;
        $cogs = $salesStats->total_cogs ?? 0;
        $returnVal = $returnsStats->total_return_value ?? 0;
        $damageVal = $damagesStats->total_damage_value ?? 0;

        $grossProfit = $revenue - $cogs;

        // Re-calculating properly:
        $returnsCostQuery = clone $returnsQuery;
        $returnsCost = $returnsCostQuery->sum(DB::raw('sales_returns.quantity_pieces * stock_batches.cost_per_piece'));

        $netRevenue = $revenue - $returnVal;
        $netCogs = $cogs - $returnsCost;
        $operatingProfit = $netRevenue - $netCogs;
        $profitAfterDamage = $operatingProfit - $damageVal;

        $finalProfit = $profitAfterDamage - $totalExpenses; // Net Realized Profit


        // ------------------------------------------------------------------
        // E. Trend Data (Charts)
        // ------------------------------------------------------------------
        // We need a daily Series for: Revenue, Profit, Pieces Sold.

        // Date formatting for Group By
        $dateFormat = match ($viewBy) {
            'year' => '%Y',
            'month' => '%Y-%m',
            default => '%Y-%m-%d'
        };

        // Sales Trend
        $salesTrend = DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('stock_batches', 'sales_order_items.batch_id', '=', 'stock_batches.id')
            ->whereBetween('sales_orders.invoice_date', [$fromDate, $toDate])
            ->where('sales_orders.status', '!=', 'CANCELLED')
            ->when($brandId && $brandId !== 'ALL', fn($q) => $q->where('stock_batches.brand_id', $brandId))
            ->when($sizeId && $sizeId !== 'ALL', fn($q) => $q->where('stock_batches.size_id', $sizeId))
            ->select(
                DB::raw("DATE_FORMAT(sales_orders.invoice_date, '$dateFormat') as date"),
                DB::raw('SUM(sales_order_items.quantity_pieces * sales_order_items.price_per_piece) as revenue'),
                DB::raw('SUM(sales_order_items.quantity_pieces * stock_batches.cost_per_piece) as cost')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->keyBy('date');

        // Merge logic for chart
        // We will build a list of dates between from/to and fill data
        // For simplicity, just return the data points we have, frontend can fill gaps or show bars.

        $trends = $salesTrend->map(function ($item) {
            return [
                'date' => $item->date,
                'revenue' => (float) $item->revenue,
                'profit' => (float) $item->revenue - (float) $item->cost, // Approx Gross Profit
            ];
        });

        // ------------------------------------------------------------------
        // F. Breakdowns (Pie Charts)
        // ------------------------------------------------------------------

        // Brand Share (Revenue)
        $brandQuery = DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('stock_batches', 'sales_order_items.batch_id', '=', 'stock_batches.id')
            ->leftJoin('brands', 'stock_batches.brand_id', '=', 'brands.id')
            ->whereBetween('sales_orders.invoice_date', [$fromDate, $toDate])
            ->where('sales_orders.status', '!=', 'CANCELLED');

        $applyFilters($brandQuery, 'stock_batches');

        $brandStats = $brandQuery->groupBy('brands.name')
            ->select(DB::raw("COALESCE(brands.name, 'No Brand') as name"), DB::raw('SUM(sales_order_items.quantity_pieces) as value'))
            ->orderByDesc('value')
            ->limit(5)
            ->get()
            ->map(function ($item) {
                $item->value = (float) $item->value;
                return $item;
            });

        // Size Share (Pieces)
        $sizeQuery = DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('stock_batches', 'sales_order_items.batch_id', '=', 'stock_batches.id')
            ->leftJoin('sizes', 'stock_batches.size_id', '=', 'sizes.id')
            ->whereBetween('sales_orders.invoice_date', [$fromDate, $toDate])
            ->where('sales_orders.status', '!=', 'CANCELLED');

        $applyFilters($sizeQuery, 'stock_batches');

        $sizeStats = $sizeQuery->groupBy('sizes.name')
            ->select(DB::raw("COALESCE(sizes.name, 'No Size') as name"), DB::raw('SUM(sales_order_items.quantity_pieces) as value'))
            ->orderByDesc('value')
            ->limit(5)
            ->get()
            ->map(function ($item) {
                $item->value = (float) $item->value;
                return $item;
            });

        // ------------------------------------------------------------------
        // G. Inventory Stats (Snapshot) - NOT filtered by date (Current State)
        // ------------------------------------------------------------------
        // We might filter by Brand/Size/Supplier though.

        $inventoryQuery = DB::table('stock_batches')
            ->where('status', '!=', 'CLOSED')
            ->where(function ($q) use ($fromDate, $toDate) {
                // Use received_date if available, else created_at
                $q->whereBetween('received_date', [$fromDate, $toDate])
                    ->orWhereNull('received_date')
                    ->whereBetween('created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"]);
            });

        $applyFilters($inventoryQuery, 'stock_batches', 'id');

        $inventoryStats = $inventoryQuery->select(
            DB::raw("SUM(CASE WHEN category = 'RAW_MATERIAL' THEN available_weight_kg ELSE 0 END) as raw_material_weight"),
            DB::raw("SUM(CASE WHEN category = 'RAW_MATERIAL' THEN available_weight_kg * cost_per_kg ELSE 0 END) as raw_material_value"),

            DB::raw("SUM(CASE WHEN category = 'FINISHED_GOOD' THEN available_pieces ELSE 0 END) as finished_goods_pieces"),
            DB::raw("SUM(CASE WHEN category = 'FINISHED_GOOD' THEN available_pieces * cost_per_piece ELSE 0 END) as finished_goods_value"),

            DB::raw("SUM(available_pieces * cost_per_piece + available_weight_kg * cost_per_kg) as total_inventory_value") // Simplified approximation
        )->first();

        return response()->json([
            'summary' => [
                'total_revenue' => (float) $revenue,
                'total_cogs' => (float) $cogs,
                'gross_profit' => (float) $grossProfit,
                'net_profit' => (float) $finalProfit,
                'total_expenses' => (float) $totalExpenses,
                'total_sold_result_pieces' => (int) ($salesStats->total_sold_pieces ?? 0),
                'total_return_value' => (float) $returnVal,
                'total_damage_value' => (float) $damageVal,
                'total_orders' => (int) ($salesStats->total_orders ?? 0)
            ],
            'inventory' => [
                'raw_material_weight' => (float) $inventoryStats->raw_material_weight,
                'raw_material_value' => (float) $inventoryStats->raw_material_value,
                'finished_goods_pieces' => (int) $inventoryStats->finished_goods_pieces,
                'finished_goods_value' => (float) $inventoryStats->finished_goods_value,
                'total_value' => (float) $inventoryStats->raw_material_value + (float) $inventoryStats->finished_goods_value
            ],
            'trends' => $trends->values(),

            'breakdowns' => [
                'brands' => $brandStats,
                'sizes' => $sizeStats
            ],
            'counts' => [
                'customers' => DB::table('customers')
                    ->whereBetween('created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"])
                    ->whereNull('deleted_at')->count(),
                'sizes' => DB::table('sizes')->count(),
                'sub_sizes' => DB::table('sub_sizes')->count(), // Static master data
                'users' => DB::table('users')
                    ->whereBetween('created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"])
                    ->count(),
                'suppliers' => DB::table('suppliers')
                    ->whereBetween('created_at', ["$fromDate 00:00:00", "$toDate 23:59:59"])
                    ->count(),
            ]
        ]);
    }
}
