<?php

use App\Http\Controllers\Api\SalesController;
use App\Http\Controllers\Api\StockController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

Route::get('/user', function (Request $request) {
    return $request->user();
})->middleware('auth:sanctum');

// Public routes for MVP (Add auth later)
Route::get('/stock/next-sequence', [StockController::class, 'getNextBatchSequence']);
Route::get('/stock/next-sequence', [StockController::class, 'getNextBatchSequence']);
Route::get('/stock/next-sequence', [StockController::class, 'getNextBatchSequence']);
Route::apiResource('stock', StockController::class);
Route::get('/settings', [\App\Http\Controllers\Api\SettingController::class, 'index']);

Route::get('/debug-dashboard', function () {
    try {
        $query = \Illuminate\Support\Facades\DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('stock_batches', 'sales_order_items.batch_id', '=', 'stock_batches.id')
            ->leftJoin('brands', 'stock_batches.brand_id', '=', 'brands.id');

        // 1. All Data (No Filters)
        $allData = (clone $query)
            ->groupBy('brands.name')
            ->select(\Illuminate\Support\Facades\DB::raw("COALESCE(brands.name, 'No Brand') as name"), \Illuminate\Support\Facades\DB::raw('SUM(sales_order_items.quantity_pieces) as value'))
            ->get();

        // 2. Filtered by "This Year" (2026)
        $filteredData = (clone $query)
            ->whereBetween('sales_orders.invoice_date', ['2026-01-01', '2026-12-31'])
            ->groupBy('brands.name')
            ->select(\Illuminate\Support\Facades\DB::raw("COALESCE(brands.name, 'No Brand') as name"), \Illuminate\Support\Facades\DB::raw('SUM(sales_order_items.quantity_pieces) as value'))
            ->get();

        return response()->json([
            'all_data' => $allData,
            'filtered_2026' => $filteredData,
            'sales_count' => \Illuminate\Support\Facades\DB::table('sales_orders')->count(),
            'items_count' => \Illuminate\Support\Facades\DB::table('sales_order_items')->count(),
        ]);
    } catch (\Exception $e) {
        return ['error' => $e->getMessage()];
    }
});

// Route::apiResource('sales', SalesController::class); // Moved to auth group to fix route conflict
Route::post('/production', [\App\Http\Controllers\Api\ProductionController::class, 'store']);

Route::post('/login', [\App\Http\Controllers\Api\AuthController::class, 'login'])->name('login');
Route::post('/logout', [\App\Http\Controllers\Api\AuthController::class, 'logout']);
Route::get('/me', [\App\Http\Controllers\Api\AuthController::class, 'me'])->middleware('auth:sanctum');
Route::post('/change-password', [\App\Http\Controllers\Api\AuthController::class, 'changePassword'])->middleware('auth:sanctum');

Route::middleware(['auth:sanctum', 'role:ADMIN'])->group(function () {
    Route::apiResource('suppliers', \App\Http\Controllers\Api\SupplierController::class);

    // Master Data Management (Write Access)
    Route::apiResource('sizes', \App\Http\Controllers\Api\SizeController::class)->except(['index', 'show']);
    Route::apiResource('sub-sizes', \App\Http\Controllers\Api\SubSizeController::class)->except(['index', 'show']);
    Route::apiResource('brands', \App\Http\Controllers\Api\BrandController::class)->except(['index', 'show']);

    // Restricted Actions
    Route::delete('/customers/{customer}', [\App\Http\Controllers\Api\CustomerController::class, 'destroy']);

    // User Management
    Route::apiResource('users', \App\Http\Controllers\Api\UserController::class);
    Route::post('/settings', [\App\Http\Controllers\Api\SettingController::class, 'update']);

    // Audit Logs
    Route::get('audit-logs', [\App\Http\Controllers\Api\AuditController::class, 'index']);
});

Route::middleware('auth:sanctum')->group(function () {
    // Read-Only Access for Master Data (Dropdowns etc.)
    Route::get('/sizes', [\App\Http\Controllers\Api\SizeController::class, 'index']);
    Route::get('/sub-sizes', [\App\Http\Controllers\Api\SubSizeController::class, 'index']);
    Route::get('/brands', [\App\Http\Controllers\Api\BrandController::class, 'index']);
    Route::get('/sub-sizes', [\App\Http\Controllers\Api\SubSizeController::class, 'index']);
    Route::get('/brands', [\App\Http\Controllers\Api\BrandController::class, 'index']);

    // Customer Management (Detailed Access, except Delete)
    Route::get('/customers', [\App\Http\Controllers\Api\CustomerController::class, 'index']);
    Route::post('/customers', [\App\Http\Controllers\Api\CustomerController::class, 'store']);
    Route::get('/customers/{customer}', [\App\Http\Controllers\Api\CustomerController::class, 'show']);
    Route::put('/customers/{customer}', [\App\Http\Controllers\Api\CustomerController::class, 'update']);
});

// Sales can be accessed by both (or add specific role if needed)
Route::middleware('auth:sanctum')->group(function () {
    Route::get('sales/payments', [SalesController::class, 'payments']);
    Route::post('sales/{id}/payment', [SalesController::class, 'updatePayment']); // Appends new payment
    Route::put('sales/payments/{id}', [SalesController::class, 'updatePaymentRecord']); // Edits existing payment
    Route::post('sales/{id}/void', [SalesController::class, 'void']);
    Route::get('sales/{id}/pdf', [SalesController::class, 'downloadPdf']);
    Route::apiResource('sales', SalesController::class);
    Route::get('sales-returns/{id}/pdf', [\App\Http\Controllers\Api\SalesReturnController::class, 'downloadPdf']);
    Route::apiResource('sales-returns', \App\Http\Controllers\Api\SalesReturnController::class);
    Route::post('sales-returns/{returnNo}/sync', [\App\Http\Controllers\Api\SalesReturnController::class, 'sync']);
    Route::post('sales-returns/{id}/void', [\App\Http\Controllers\Api\SalesReturnController::class, 'void']);

    Route::apiResource('sales-damages', \App\Http\Controllers\Api\SalesDamageController::class);
    Route::post('sales-damages/{id}/void', [\App\Http\Controllers\Api\SalesDamageController::class, 'void']);

    // Dashboard Stats
    Route::get('/dashboard/stats', [\App\Http\Controllers\Api\DashboardController::class, 'stats']);

    // Bank Accounts
    Route::post('bank-accounts/{id}/default', [\App\Http\Controllers\Api\BankAccountController::class, 'makeDefault']);
    Route::apiResource('bank-accounts', \App\Http\Controllers\Api\BankAccountController::class);
    Route::apiResource('expenses', \App\Http\Controllers\Api\ExpenseController::class);
    Route::post('expenses/{expenseNo}/void', [\App\Http\Controllers\Api\ExpenseController::class, 'void']);
    Route::get('expenses/{expenseNo}/items', [\App\Http\Controllers\Api\ExpenseController::class, 'items']);
    Route::put('expenses/{expenseNo}/batch', [\App\Http\Controllers\Api\ExpenseController::class, 'updateBatch']);
    Route::apiResource('expense-categories', \App\Http\Controllers\Api\ExpenseCategoryController::class);

    Route::get('/production', [\App\Http\Controllers\Api\ProductionController::class, 'index']);
    Route::post('/production', [\App\Http\Controllers\Api\ProductionController::class, 'store']);
    Route::delete('/production/{id}', [\App\Http\Controllers\Api\ProductionController::class, 'destroy']);
    Route::put('/production/{id}', [\App\Http\Controllers\Api\ProductionController::class, 'update']);
    Route::get('/production/{id}', [\App\Http\Controllers\Api\ProductionController::class, 'show']);
    Route::get('/production/{id}', [\App\Http\Controllers\Api\ProductionController::class, 'show']);
    Route::post('/stock/{id}/void', [StockController::class, 'void']);
    Route::apiResource('stock', StockController::class); // Stock view needed for sales
});

// Test Broadcast Route
Route::get('/test-broadcast', function () {
    broadcast(new \App\Events\StockUpdated(\App\Models\StockBatch::first()));
    return 'Broadcast sent';
});
